﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Net;

namespace Argos_3D_P100_Visualizer
{
    public class PmdSdk
    {
        [DllImport("pmdaccess2.dll")]
        static extern int pmdOpen(ref uint hnd, [MarshalAs(UnmanagedType.LPStr)] string rplugin, [MarshalAs(UnmanagedType.LPStr)] string rparam, [MarshalAs(UnmanagedType.LPStr)] string pplugin, [MarshalAs(UnmanagedType.LPStr)] string pparam);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdClose(uint hnd);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdUpdate(uint hnd);
        
        [DllImport("pmdaccess2.dll")]
        static extern int pmdSetIntegrationTime(uint hnd, UInt32 idx, UInt32 t);
        
        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetValidIntegrationTime(uint hnd, ref UInt32 result, UInt32 idx, ProximityENUM w, UInt32 t);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdSetModulationFrequency(uint hnd, UInt32 idx, UInt32 t);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetValidModulationFrequency(uint hnd, ref UInt32 result, UInt32 idx, ProximityENUM w, UInt32 t);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetSourceDataDescription(uint hnd, ref PMDDataDescription dd);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetSourceDataSize(uint hnd, ref UInt32 size);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetSourceData(uint hnd, [MarshalAs(UnmanagedType.LPArray)] UInt32[] data, UInt32 maxLen);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdCalcDistances(uint hnd, [MarshalAs(UnmanagedType.LPArray)] float[] data, UInt32 maxLen, PMDDataDescription sourceDD, uint[] sourceData);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdCalcAmplitudes(uint hnd, [MarshalAs(UnmanagedType.LPArray)] float[] data, UInt32 maxLen, PMDDataDescription sourceDD, uint[] sourceData);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdCalcFlags(uint hnd, [MarshalAs(UnmanagedType.LPArray)] UInt32[] data, UInt32 maxLen, PMDDataDescription sourceDD, uint[] sourceData);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetDistances(uint hnd, [MarshalAs(UnmanagedType.LPArray)] float[] data, UInt32 maxLen);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetAmplitudes(uint hnd, [MarshalAs(UnmanagedType.LPArray)] float[] data, UInt32 maxLen);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetFlags(uint hnd, [MarshalAs(UnmanagedType.LPArray)] UInt32[] data, UInt32 maxLen);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGet3DCoordinates(uint hnd, [MarshalAs(UnmanagedType.LPArray)] float[] data, UInt32 maxLen);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdGetLastError(uint hnd, [MarshalAs(UnmanagedType.LPArray)] byte[] answer, UInt32 maxLen);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdProcessingCommand(uint hnd, [MarshalAs(UnmanagedType.LPArray)] byte[] result, UInt32 maxLen, [MarshalAs(UnmanagedType.LPStr)] string cmd);

        [DllImport("pmdaccess2.dll")]
        static extern int pmdSourceCommand(uint hnd, [MarshalAs(UnmanagedType.LPArray)] byte[] result, UInt32 maxLen, [MarshalAs(UnmanagedType.LPStr)] string cmd);

        #region PrivateDeclarations
        private uint hndl = 0;
        private string statusMsg = "";
        private static string[] deviceTypes = new string[] { "unused", "150-2001-1", "160-0001-1", "160-0001-2" };
        #endregion

        public int Xres { get; private set; }
        public int Yres { get; private set; }


        static bool inUse = false;

        public PmdSdk(string srcPlugin, string srcParam, string procPlugin, string procParam)
        {
            lock (this)
            {
                if (inUse)
                {
                    throw new Exception("PmdSdk is already in use");
                }
                inUse = true;
                hndl = 0;

                int error_result = pmdOpen(ref hndl, srcPlugin, srcParam, procPlugin, procParam);
                if (error_result != 0)
                {
                    inUse = false;
                    statusMsg = "ERROR: open failed";
                    throw new Exception("PmdSdk Open failed! " + getErrorMsg(error_result) + "!\n");
                }
                statusMsg = "Successfully opened handle " + hndl.ToString();
            }
        }


        public void Close()
        {
            lock (this)
            {
                int error_result = pmdClose(hndl);
                if (error_result != 0)
                {
                    statusMsg = "ERROR: close failed";
                    throw new Exception("PmdSdk Close failed! " + getErrorMsg(error_result) + "!\n");
                }
                statusMsg = "Successfully closed handle " + hndl.ToString();
                hndl = 0;
                inUse = false;
            }
        }


        public uint getHandle()
        {
            return hndl;
        }
        
        private String getLastError()
        {
            lock (this)
            {
                byte[] answer = new byte[128];
                int error_result = pmdGetLastError(hndl, answer, 128);
                if (error_result == 0)
                {
                    return (new ASCIIEncoding().GetString(answer)).Split('\0')[0];
                }
                return "[Can't read error description from PmdSdk]";
            }
        }



        private String getErrorMsg(int error)
        {
            switch (error)
            {
                case 1024:
                    { return "PMD runtime error"; }
                case 1027:
                    { return "PMD invalid value"; }
                case 1028:
                    { return "PMD timeout"; }
                case 2050:
                    { return "PMD not implemented"; }
            }
            return error.ToString();
        }


        public void execSourceCmd(string cmd)
        {
            lock (this)
            {
                byte[] answer = new byte[512];
                int error_result = pmdSourceCommand(hndl, answer, 512, cmd);
                statusMsg = (new ASCIIEncoding().GetString(answer)).Split('\0')[0] + " (" + cmd + ")";
                if (error_result != 0)
                {
                    throw new Exception(cmd + " failed! " + getErrorMsg(error_result) + "!\n" + statusMsg);
                }
            }
        }


        public void execProcessingCmd(string cmd)
        {
            lock (this)
            {
                byte[] answer = new byte[512];
                int error_result = pmdProcessingCommand(hndl, answer, 512, cmd);
                statusMsg = (new ASCIIEncoding().GetString(answer)).Split('\0')[0] + " (" + cmd + ")";
                if (error_result != 0)
                {
                    throw new Exception(cmd + " failed! " + getErrorMsg(error_result) + "!\n" + statusMsg);
                }
            }
        }


        public void setRegister(AddressRegistersPmdSdk addr, Int64 value)
        {
            if (addr == AddressRegistersPmdSdk.ImageDataFormat)
            {
                ImageMode = (ImageDataFormatModes) value;
                return;
            }
            if (addr == AddressRegistersPmdSdk.FilterConfig)
            {
                FilterConfig = Convert.ToUInt32(value);
                return;
            }
            if (addr == AddressRegistersPmdSdk.IsCalibrationDataLoaded)
            {
                throw new Exception(statusMsg = "IsCalibrationDataLoaded is read only");
            }

            int error_result;
            int sequence = 0;
            String cmd = "";
            lock (this)
            {
                statusMsg = "result pending..";
                if (addr >= AddressRegistersPmdSdk.Sequence0PllSelect && addr < AddressRegistersPmdSdk.PhysicalAddressBoundary)
                {
                    // check for sequence registers which are repeated in steps of 10
                    sequence = (addr - AddressRegistersPmdSdk.Sequence0PllSelect) / 10;
                    if (addr - sequence*10 == AddressRegistersPmdSdk.Sequence0IntTime)
                    {
                        SetIntegrationTime(value, sequence);
                        return;
                    }
                    else if (addr - sequence * 10 == AddressRegistersPmdSdk.Sequence0ModFrequency)
                    {
                        SetModulationFrequency(value, sequence);
                        return;
                    }
                    else if (addr - sequence * 10 == AddressRegistersPmdSdk.Sequence0DistanceOffset)
                    {
                        cmd = "SetOffset " + sequence + " " + value.ToString();
                    }
                    else
                    {
                        cmd = "SetRegister " + ((int)addr).ToString() + " " + value.ToString();
                    }
                }

                else if (addr == AddressRegistersPmdSdk.LoadRegisterMap)
                {
                    cmd = "LoadRegisterMap " + _filenameParameter;
                    _filenameParameter = null;
                }
                else if (addr == AddressRegistersPmdSdk.WriteRegisterMap)
                {
                    cmd = "WriteRegisterMap " + _filenameParameter;
                    _filenameParameter = null;
                }
                else if (addr == AddressRegistersPmdSdk.LoadFirmware)
                {
                    cmd = "LoadFirmware " + _filenameParameter;
                    _filenameParameter = null;
                }
                else if (addr == AddressRegistersPmdSdk.WriteConfigToFlash)
                {
                    cmd = "WriteConfigToFlash";
                }
                else if (addr == AddressRegistersPmdSdk.WriteFirmwareToFlash)
                {
                    cmd = "WriteFirmwareToFlash";
                }
                else if (addr == AddressRegistersPmdSdk.DisablelntegrationTimeCheck)
                {
                    cmd = "DisableIntegrationTimeCheck";
                }
                else if (addr == AddressRegistersPmdSdk.LoadCalibrationData)
                {
                    cmd = "LoadCalibrationData " + _filenameParameter;
                    _filenameParameter = null;
                }
                else if (addr == AddressRegistersPmdSdk.SaveCalibrationData)
                {
                    cmd = "SaveCalibrationData " + _filenameParameter;
                    _filenameParameter = null;
                }
                else if (addr == AddressRegistersPmdSdk.NoSerialCustomer)
                {
                    cmd = "StoreSerialNumber " + value.ToString();
                }
                else if ((int)addr >= 0 && (int)addr < (int)AddressRegistersPmdSdk.PhysicalAddressBoundary)
                {
                    // the general case: write value directly into register
                    cmd = "SetRegister " + ((int)addr).ToString() + " " + value.ToString();
                }
                else
                {
                    throw new Exception(statusMsg = "Register " + addr + " not implemented");
                }

                byte[] answer = new byte[128];
                error_result = pmdSourceCommand(hndl, answer, 128, cmd);
                statusMsg = (new ASCIIEncoding().GetString(answer)).Split('\0')[0] + " (" + cmd + ")";
                if (error_result != 0)
                {
                    throw new Exception(cmd + " failed! " + getErrorMsg(error_result) + "!\n" + statusMsg);
                }
            }
        }

        
        public Int64 getRegister(AddressRegistersPmdSdk addr)
        {
            if (addr == AddressRegistersPmdSdk.ImageDataFormat)
            {
                return (Int64) ImageMode;
            }
            if (addr == AddressRegistersPmdSdk.FilterConfig)
            {
                return FilterConfig;
            }
            if (addr == AddressRegistersPmdSdk.LoadRegisterMap)
            {
                throw new Exception(statusMsg = "LoadRegisterMap is write only");
            }
            if (addr == AddressRegistersPmdSdk.WriteRegisterMap)
            {
                throw new Exception(statusMsg = "WriteRegisterMap is write only");
            }
            if (addr == AddressRegistersPmdSdk.WriteConfigToFlash)
            {
                throw new Exception(statusMsg = "WriteConfigToFlash is write only");
            }
            if (addr == AddressRegistersPmdSdk.WriteFirmwareToFlash)
            {
                throw new Exception(statusMsg = "WriteFirmwareToFlash is write only");
            }
            if (addr == AddressRegistersPmdSdk.DisablelntegrationTimeCheck)
            {
                throw new Exception(statusMsg = "DisablelntegrationTimeCheck is write only");
            }
            if (addr == AddressRegistersPmdSdk.LoadCalibrationData)
            {
                throw new Exception(statusMsg = "LoadCalibrationData is write only");
            }
            if (addr == AddressRegistersPmdSdk.LoadFirmware)
            {
                throw new Exception(statusMsg = "LoadFirmware is write only");
            }
            if (addr == AddressRegistersPmdSdk.IsCalibrationDataLoaded)
            {
                return IsCalibrationDataLoaded;
            }
            if (addr == AddressRegistersPmdSdk.SaveCalibrationData)
            {
                throw new Exception(statusMsg = "SaveCalibrationData is write only");
            }

            int sequence = 0;
            string cmd = "";
            if (addr >= AddressRegistersPmdSdk.Sequence0PllSelect && addr < AddressRegistersPmdSdk.PhysicalAddressBoundary)
            {
                // check for sequence registers which are repeated in steps of 10
                sequence = (addr - AddressRegistersPmdSdk.Sequence0PllSelect) / 10;
                if (addr - sequence * 10 == AddressRegistersPmdSdk.Sequence0DistanceOffset)
                {
                    cmd = "GetOffset " + sequence + " ";
                }
                else
                {
                    cmd = "GetRegister " + ((int)addr).ToString();
                }
            }
            else if ((int)addr >= 0 && (int)addr < (int)AddressRegistersPmdSdk.PhysicalAddressBoundary)
            {
                cmd = "GetRegister " + ((int)addr).ToString();
            }
            else
            {
                throw new Exception(statusMsg = "Register " + addr + " not implemented");
            }
            lock (this)
            {
                statusMsg = "result pending..";
                byte[] answer = new byte[128];
                int error_result = pmdSourceCommand(hndl, answer, 128, cmd);
                statusMsg = (new ASCIIEncoding().GetString(answer)).Split('\0')[0];
                if (error_result != 0)
                {
                    throw new Exception(statusMsg = "Error " + statusMsg + " (PmdSdk GetRegister " + addr.ToString() + " failed! " + getErrorMsg(error_result) + "!)");
                }
                try
                {
                    UInt32 i = Convert.ToUInt32(statusMsg, 10);
                    statusMsg += " (" + cmd + ")";
                    return i;
                }
                catch
                {
                }
                try
                {
                    UInt32 i = Convert.ToUInt32(statusMsg, 16);
                    statusMsg += " (" + cmd + ")";
                    return i;
                }
                catch
                {
                }
                throw new Exception(statusMsg = "Could not parse: " + statusMsg);
            }
        }


        public PMDDataDescription Update()
        {
            lock (this)
            {
                PMDDataDescription descr = new PMDDataDescription();
                descr.img = new PMDImageData();
                int error_result = pmdUpdate(hndl);
                if (error_result != 0)
                {
                    Xres = 0;
                    Yres = 0;
                    throw new Exception("Update failed! " + getErrorMsg(error_result) + "!\n" + getLastError());
                }
                error_result = pmdGetSourceDataDescription(hndl, ref descr);
                if (error_result != 0)
                {
                    Xres = 0;
                    Yres = 0;
                    throw new Exception("pmdGetSourceDataDescription failed! " + getErrorMsg(error_result) + "!\n" + getLastError());
                }
                Xres = Convert.ToInt32(descr.img.numColumns);
                Yres = Convert.ToInt32(descr.img.numRows);
                return descr;
            }
        }


        public uint[] getSourceData()
        {
            lock (this)
            {
                UInt32 sizeData = 0;
                pmdGetSourceDataSize(hndl, ref sizeData);
                UInt32[] data32 = new UInt32[sizeData];
                int error_result = pmdGetSourceData(hndl, data32, sizeData);
                if (error_result == 0)
                {
                    return data32;
                }
            }
            return new uint[0];
        }


        public List<ContainerSTRUCT> getData(SourceDataSTRUCT sourceData, bool invalidateByFlags)
        {
            if (sourceData.descr.img.numColumns > 0 && sourceData.descr.img.numRows > 0)
            {
                ContainerSTRUCT containerStruct0 = new ContainerSTRUCT();
                ContainerSTRUCT containerStruct1 = new ContainerSTRUCT();
                containerStruct0.header = new SortedList<ContainerHeaderENUM, UInt32>();
                if (sourceData.sourceData.Length >= 128)
                {
                    foreach (ContainerHeaderENUM reg in Enum.GetValues(typeof(ContainerHeaderENUM)))
                    {
                        containerStruct0.header[reg] = (UInt32)IPAddress.NetworkToHostOrder((int)sourceData.sourceData[(int)reg]);
                    }
                }
                containerStruct1.header = new SortedList<ContainerHeaderENUM, UInt32>();
                if (sourceData.sourceData.Length >= Xres * Yres / 2 + 128 + 128)
                {
                    foreach (ContainerHeaderENUM reg in Enum.GetValues(typeof(ContainerHeaderENUM)))
                    {
                        containerStruct1.header[reg] = (UInt32)IPAddress.NetworkToHostOrder((int)sourceData.sourceData[Xres * Yres / 2 + 128 + (int)reg]);
                    }
                }
                float[] dist = new float[sourceData.descr.img.numColumns * sourceData.descr.img.numRows];
                float[] amp = new float[sourceData.descr.img.numColumns * sourceData.descr.img.numRows];
                int error_result_dist = pmdCalcDistances(hndl, dist, Convert.ToUInt32(dist.Length * sizeof(float)), sourceData.descr, sourceData.sourceData);
                int error_result_amp = pmdCalcAmplitudes(hndl, amp, Convert.ToUInt32(amp.Length * sizeof(float)), sourceData.descr, sourceData.sourceData);
                if (error_result_dist == 0 && error_result_amp == 0)
                {
                    containerStruct0.data = new List<double>();
                    containerStruct1.data = new List<double>();
                    if (invalidateByFlags)
                    {
                        uint[] flags = new uint[sourceData.descr.img.numColumns * sourceData.descr.img.numRows];
                        int error_result_flags = pmdCalcFlags(hndl, flags, Convert.ToUInt32(flags.Length * sizeof(float)), sourceData.descr, sourceData.sourceData);
                        if (error_result_flags == 0)
                        {
                            for (int i = 0; i < dist.Length; i++)
                            {
                                if (invalidateByFlags && ((flags[i] & PMD_FLAG_INVALID) > 0 || (flags[i] & PMD_FLAG_SATURATED) > 0 || (flags[i] & PMD_FLAG_INCONSISTENT) > 0 || (flags[i] & PMD_FLAG_LOW_SIGNAL) > 0))
                                {
                                    containerStruct0.data.Add(0);
                                    containerStruct1.data.Add(0);
                                }
                                else
                                {
                                    containerStruct0.data.Add(Convert.ToDouble(dist[i]) * 1000);
                                    containerStruct1.data.Add(Convert.ToDouble(amp[i]));
                                }
                            }
                        }
                    }
                    else
                    {
                        for (int i = 0; i < dist.Length; i++)
                        {
                            containerStruct0.data.Add(Convert.ToDouble(dist[i]) * 1000);
                            containerStruct1.data.Add(Convert.ToDouble(amp[i]));
                        }
                    }
                }
                List<ContainerSTRUCT> data = new List<ContainerSTRUCT>();
                data.Add(containerStruct0);
                data.Add(containerStruct1);
                return data;
            }
            return new List<ContainerSTRUCT>();
        }


        //public List<ContainerSTRUCT> getData(bool invalidateByFlags)
        //{
        //    if (((int)_imageMode & 0xfff8) == (int)ImageDataFormatModes.DistAmp)
        //    {
        //        List<ContainerSTRUCT> data = new List<ContainerSTRUCT>();
        //        ContainerSTRUCT containerStruct0 = new ContainerSTRUCT();
        //        containerStruct0.data = GetDistances(invalidateByFlags);
        //        data.Add(containerStruct0);
        //        ContainerSTRUCT containerStruct1 = new ContainerSTRUCT();
        //        containerStruct1.data = GetAmplitudes();
        //        data.Add(containerStruct1);
        //        return data;
        //    }

        //    if (((int)_imageMode & 0xfff8) == (int)ImageDataFormatModes.DistAmpSourceData)
        //    {
        //        lock (this)
        //        {
        //            UInt32 sizeData = 0;
        //            pmdGetSourceDataSize(_hndl, ref sizeData);
        //            UInt32[] data32 = new UInt32[sizeData];
        //            UInt32[] header0 = new UInt32[128];
        //            UInt32[] header1 = new UInt32[128];
        //            UInt32[] header2 = new UInt32[128];
        //            //UInt32[] header3 = new UInt32[128];
        //            int error_result = pmdGetSourceData(_hndl, data32, sizeData);
        //            if (error_result == 0)
        //            {
        //                for (int i = 0; i < 128; i++)
        //                {
        //                    header0[i] = (UInt32)IPAddress.NetworkToHostOrder((int)data32[i]);
        //                    header1[i] = (UInt32)IPAddress.NetworkToHostOrder((int)data32[Xres * Yres / 2 + 128 + i]);
        //                    header2[i] = (UInt32)IPAddress.NetworkToHostOrder((int)data32[2 * (Xres * Yres / 2 + 128) + i]);
        //                    //header3[i] = (UInt32)IPAddress.NetworkToHostOrder((int)data32[i]);
        //                }
        //                List<double> data0 = new List<double>();
        //                List<double> data1 = new List<double>();
        //                List<double> data2 = new List<double>();
        //                //List<double> data3 = new List<double>();
        //                for (int i = 0; i < Xres * Yres / 2; i++)
        //                {
        //                    UInt32 tempData0 = data32[i + 128];
        //                    UInt32 tempData1 = data32[Xres * Yres / 2 + 128 + i + 128];
        //                    UInt32 tempData2 = data32[2 * (Xres * Yres / 2 + 128) + i + 128];
        //                    data0.Add(Convert.ToDouble(tempData0 & 0xffff));
        //                    data0.Add(Convert.ToDouble((tempData0 & 0xffff0000) >> 16));
        //                    data1.Add(Convert.ToDouble(tempData1 & 0xffff));
        //                    data1.Add(Convert.ToDouble((tempData1 & 0xffff0000) >> 16));
        //                    data2.Add(Convert.ToDouble(tempData2 & 0xffff));
        //                    data2.Add(Convert.ToDouble((tempData2 & 0xffff0000) >> 16));
        //                    //data3.Add(Convert.ToDouble(tempData3 & 0xffff));
        //                    //data3.Add(Convert.ToDouble((tempData3 & 0xffff0000) >> 16));
        //                }
        //                List<ContainerSTRUCT> data = new List<ContainerSTRUCT>();
        //                ContainerSTRUCT containerStruct0 = new ContainerSTRUCT();
        //                containerStruct0.header = new SortedList<ContainerHeaderENUM, uint>();
        //                foreach (ContainerHeaderENUM reg in Enum.GetValues(typeof(ContainerHeaderENUM)))
        //                {
        //                    containerStruct0.header.Add(reg, header0[(int)reg]);
        //                }
        //                containerStruct0.data = data0;
        //                data.Add(containerStruct0);
        //                ContainerSTRUCT containerStruct1 = new ContainerSTRUCT();
        //                containerStruct1.header = new SortedList<ContainerHeaderENUM, uint>();
        //                foreach (ContainerHeaderENUM reg in Enum.GetValues(typeof(ContainerHeaderENUM)))
        //                {
        //                    containerStruct1.header.Add(reg, header1[(int)reg]);
        //                }
        //                containerStruct1.data = data1;
        //                data.Add(containerStruct1);
        //                ContainerSTRUCT containerStruct2 = new ContainerSTRUCT();
        //                containerStruct2.header = new SortedList<ContainerHeaderENUM, uint>();
        //                foreach (ContainerHeaderENUM reg in Enum.GetValues(typeof(ContainerHeaderENUM)))
        //                {
        //                    containerStruct2.header.Add(reg, header2[(int)reg]);
        //                }
        //                containerStruct2.data = data2;
        //                data.Add(containerStruct2);
        //                //ContainerSTRUCT containerStruct3 = new ContainerSTRUCT();
        //                //containerStruct3.header = new SortedList<ContainerHeaderENUM, uint>();
        //                //foreach (ContainerHeaderENUM reg in Enum.GetValues(typeof(ContainerHeaderENUM)))
        //                //{
        //                //    containerStruct3.header.Add(reg, header3[(int)reg]);
        //                //}
        //                //containerStruct3.data = data3;
        //                //data.Add(containerStruct3);
        //                return data;
        //            }
        //        }
        //    }

        //    else if (((int)_imageMode & 0xfff8) == (int)ImageDataFormatModes.XYZcoordinates)
        //    {
        //        List<ContainerSTRUCT> data = new List<ContainerSTRUCT>();
        //        lock (this)
        //        {
        //            if (Xres > 0 && Yres > 0)
        //            {
        //                float[] coords = new float[Xres * Yres * 3];
        //                int error_result = pmdGet3DCoordinates(_hndl, coords, Convert.ToUInt32(coords.Length * sizeof(float)));
        //                if (error_result == 0)
        //                {
        //                    uint[] flags = new uint[Xres * Yres];
        //                    if (invalidateByFlags)
        //                    {
        //                        error_result = pmdGetFlags(_hndl, flags, Convert.ToUInt32(flags.Length * sizeof(UInt32)));
        //                    }
        //                    if (error_result == 0)
        //                    {
        //                        List<double> coordsX = new List<double>();
        //                        List<double> coordsY = new List<double>();
        //                        List<double> coordsZ = new List<double>();
        //                        for (int i = 0; i < coords.Length; )
        //                        {
        //                            if (invalidateByFlags && ((flags[i / 3] & PMD_FLAG_INVALID) > 0 || (flags[i / 3] & PMD_FLAG_SATURATED) > 0 || (flags[i / 3] & PMD_FLAG_INCONSISTENT) > 0 || (flags[i / 3] & PMD_FLAG_LOW_SIGNAL) > 0))
        //                            {
        //                                coordsY.Add(0);
        //                                coordsZ.Add(0);
        //                                coordsX.Add(0);
        //                                i += 3;
        //                            }
        //                            else
        //                            {
        //                                coordsY.Add(Convert.ToDouble(coords[i++] * 1000));
        //                                coordsZ.Add(Convert.ToDouble(-coords[i++] * 1000));
        //                                coordsX.Add(Convert.ToDouble(coords[i++] * 1000));
        //                            }
        //                        }
        //                        ContainerSTRUCT containerStruct0 = new ContainerSTRUCT();
        //                        containerStruct0.data = coordsX;
        //                        data.Add(containerStruct0);
        //                        ContainerSTRUCT containerStruct1 = new ContainerSTRUCT();
        //                        containerStruct1.data = coordsY;
        //                        data.Add(containerStruct1);
        //                        ContainerSTRUCT containerStruct2 = new ContainerSTRUCT();
        //                        containerStruct2.data = coordsZ;
        //                        data.Add(containerStruct2);
        //                    }
        //                }
        //            }
        //        }
        //        return data;
        //    }

        //    return new List<ContainerSTRUCT>();
        //}


        //private List<double> GetDistances(bool invalidateByFlags)
        //{
        //    lock (this)
        //    {
        //        List<double> distD = new List<double>();
        //        if (Xres > 0 && Yres > 0)
        //        {
        //            float[] dist = new float[Xres * Yres];
        //            int error_result = pmdGetDistances(_hndl, dist, Convert.ToUInt32(dist.Length * sizeof(float)));
        //            if (error_result == 0)
        //            {
        //                uint[] flags = new uint[Xres * Yres];
        //                if (invalidateByFlags)
        //                {
        //                    error_result = pmdGetFlags(_hndl, flags, Convert.ToUInt32(flags.Length * sizeof(UInt32)));
        //                }
        //                if (error_result == 0)
        //                {
        //                    int i = 0;
        //                    foreach (float f in dist)
        //                    {
        //                        if (invalidateByFlags && ((flags[i] & PMD_FLAG_INVALID) > 0 || (flags[i] & PMD_FLAG_SATURATED) > 0 || (flags[i] & PMD_FLAG_INCONSISTENT) > 0 || (flags[i] & PMD_FLAG_LOW_SIGNAL) > 0))
        //                        {
        //                            distD.Add(0);
        //                        }
        //                        else
        //                        {
        //                            distD.Add(Convert.ToDouble(f) * 1000);
        //                        }
        //                        i++;
        //                    }
        //                }
        //            }
        //        }
        //        return distD;
        //    }
        //}

        //private List<double> GetAmplitudes()
        //{
        //    lock (this)
        //    {
        //        List<double> ampD = new List<double>();
        //        if (Xres > 0 && Yres > 0)
        //        {
        //            float[] amp = new float[Xres * Yres];
        //            int error_result = pmdGetAmplitudes(_hndl, amp, Convert.ToUInt32(amp.Length * sizeof(float)));
        //            if (error_result == 0)
        //            {
        //                foreach (float f in amp)
        //                {
        //                    ampD.Add(Convert.ToDouble(f));
        //                }
        //            }
        //        }
        //        return ampD;
        //    }
        //}


        //private List<uint> GetFlags()
        //{
        //    lock (this)
        //    {
        //        List<uint> flagsD = new List<uint>();
        //        if (Xres > 0 && Yres > 0)
        //        {
        //            UInt32[] flags = new UInt32[Xres * Yres];
        //            int error_result = pmdGetFlags(_hndl, flags, Convert.ToUInt32(flags.Length * sizeof(UInt32)));
        //            if (error_result == 0)
        //            {
        //                foreach (uint flag in flags)
        //                {
        //                    flagsD.Add(flag);
        //                }
        //            }
        //        }
        //        return flagsD;
        //    }
        //}


        // lock BEFORE calling me
        public void SetIntegrationTime(Int64 t_int, int sequence)
        {
            UInt32 valid_time = 0;
            int error_result = pmdGetValidIntegrationTime(hndl, ref valid_time, (UInt32)sequence, ProximityENUM.CloseTo, Convert.ToUInt32(t_int));
            if (error_result != 0)
            {
                throw new Exception("PmdSdk t_int validation failed! " + getErrorMsg(error_result) + "!\n" + getLastError());
            }
            error_result = pmdSetIntegrationTime(hndl, (UInt32)sequence, valid_time);
            if (error_result != 0)
            {
                throw new Exception(statusMsg = "Error " + getErrorMsg(error_result) + ": Integration time set ERROR: " + sequence.ToString() + ", " + valid_time.ToString());
            }
            statusMsg = "Integration time " + sequence.ToString() + " set to " + valid_time.ToString();
        }


        //public int GetIntegrationTime()
        //{
        //    uint time = 0;
        //    int error_result = pmdGetIntegrationTime(Hndl, ref time, 0);
        //    if (error_result != 0)
        //    {
        //        _lastError = error_result;
        //        throw new Exception("PmdSdk t_int read failed! " + getErrorMsg(_lastError) + "!\n" + getLastErrorMsg());
        //    }
        //    return Convert.ToInt32(time);
        //}


        // lock BEFORE calling me
        public void SetModulationFrequency(Int64 mod_freq, int sequence)
        {
            UInt32 valid_freq = 0;
            int error_result = pmdGetValidModulationFrequency(hndl, ref valid_freq, (UInt32)sequence, ProximityENUM.CloseTo, Convert.ToUInt32(mod_freq));
            if (error_result != 0)
            {
                throw new Exception("PmdSdk mod_freq validation failed! " + getErrorMsg(error_result) + "!\n" + getLastError());
            }
            error_result = pmdSetModulationFrequency(hndl, (UInt32)sequence, valid_freq);
            if (error_result != 0)
            {
                throw new Exception(statusMsg = "Error " + getErrorMsg(error_result) + ": Modulation frequency set ERROR: " + sequence.ToString() + ", " + valid_freq.ToString());
            }
            statusMsg = "Modulation frequency " + sequence.ToString() + " set to " + valid_freq.ToString();
        }



        public Int64 IsCalibrationDataLoaded
        {
            get
            {
                lock (this)
                {
                    string cmd = "IsCalibrationDataLoaded";
                    byte[] answer = new byte[128];
                    int error_result = pmdSourceCommand(hndl, answer, 128, cmd);
                    statusMsg = (new ASCIIEncoding().GetString(answer)).Split('\0')[0];
                    if (error_result != 0)
                    {
                        statusMsg = getLastError();
                        throw new Exception(cmd + " failed! " + getErrorMsg(error_result) + "!\n" + statusMsg);
                    }
                    if (statusMsg == "YES")
                    {
                        statusMsg += " (" + cmd + ")";
                        return 1;
                    }
                    if (statusMsg == "NO")
                    {
                        statusMsg += " (" + cmd + ")";
                        return 0;
                    }
                    throw new Exception(cmd + ": invalid answer! " + getErrorMsg(error_result) + "!\n" + statusMsg);
                }
            }
        }


        public string getDevicePonSn()
        {
            UInt32 sn = (UInt32)getRegister(AddressRegistersPmdSdk.NoSerialCustomer);
            UInt32 device_code = (sn & 0xfff00000) >> 20;
            if (device_code < deviceTypes.Length)
            {
                return "Argos3D - P100, S/N " + deviceTypes[device_code] + "-" + (sn &0xfffff).ToString().PadLeft(6, '0');
            }
            return "Unknown device";
        }


        private ImageDataFormatModes _imageMode;
        public ImageDataFormatModes ImageMode
        {
            get
            {
                return (ImageDataFormatModes)((int)_imageMode | 0x6);
            }
            set
            {
                _imageMode = (ImageDataFormatModes)((int)value | 0x6);
            }
        }

        private Int64 _filterConfig = 8;
        public Int64 FilterConfig
        {
            get { return _filterConfig; }
            set {
                if (value != _filterConfig)
                {
                    lock (this)
                    {
                        _filterConfig = value;
                        if ((_filterConfig & (1 << 3)) > 0)
                        {
                            byte[] answer = new byte[128];
                            int error_result = pmdProcessingCommand(hndl, answer, 128, "SetBilateralFilter on");
                            statusMsg = (new ASCIIEncoding().GetString(answer)).Split('\0')[0] + " (SetBilateralFilter on)";
                            if (error_result != 0)
                            {
                                throw new Exception(statusMsg = statusMsg + " (PmdSdk SetBilateralFilter failed! " + getErrorMsg(error_result) + "!)");
                            }
                        }
                        else
                        {
                            byte[] answer = new byte[128];
                            int error_result = pmdProcessingCommand(hndl, answer, 128, "SetBilateralFilter off");
                            statusMsg = (new ASCIIEncoding().GetString(answer)).Split('\0')[0] + " (SetBilateralFilter off)";
                            if (error_result != 0)
                            {
                                throw new Exception(statusMsg = statusMsg + " (PmdSdk SetBilateralFilter failed! " + getErrorMsg(error_result) + "!)");
                            }
                        }
                    }
                }
            }
        }

        public string getStatusMsg()
        {
            return statusMsg;
        }

        private string _filenameParameter = null;
        public string FilenameParameter
        {
            set
            {
                _filenameParameter = value;
            }
            get
            {
                return _filenameParameter;
            }
        }


        public bool Connected
        {
            get
            {
                if (hndl == 0)
                {
                    return false;
                }
                return true;
            }
        }


        public struct PMDGenericData
        {
          public uint subType;
          public uint numElem;
          public uint sizeOfElem;
        };


        public struct PMDImageData
        {
            public uint subType;
            public uint numColumns;
            public uint numRows;
            public uint numSubImages;

            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            public int[] integrationTime;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            public int[] modulationFrequency;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
            public int[] offset;

            public int pixelAspectRatio;
            public int pixelOrigin;


            public uint timeStampHi;
            public uint timeStampLo;

            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 24)]
            public char[] reserved;

            public uint userData0;
        };


        public struct PMDDataDescription
        {
            public uint PID;
            public uint DID;
            public uint type;
            public uint size;

            public uint subHeaderType;

            public PMDImageData img;
        };


        enum ProximityENUM
        {
            AtLeast, AtMost, CloseTo
        };

        // Flags
        private const uint PMD_FLAG_INVALID = 0x01;
        private const uint PMD_FLAG_SATURATED = 0x02;
        private const uint PMD_FLAG_INCONSISTENT = 0x04;
        private const uint PMD_FLAG_LOW_SIGNAL = 0x08;
        private const uint PMD_FLAG_SBI_ACTIVE = 0x10;


        enum TriggerModeENUM
        {
            Freerun, Hard, Soft
        };

        [StructLayout(LayoutKind.Explicit)]
        struct Union
        {
            [FieldOffset(0)]
            public PMDGenericData gen;
            [FieldOffset(0)]
            public PMDImageData std;
            [FieldOffset(0)]
            public PMDImageData img;
        };

        
        public enum ContainerHeaderENUM
        {
            PMDStatus,
            NoSerialCustomer,
            NoReleaseDate,
            NoFramebytes,
            NoRows,
            NoColumns,
            SetROI,
            ROIColumnBegin,
            ROIColumnEnd,
            ROIRowBegin,
            ROIRowEnd,
            SequenceLength,
            SequenceCombine,
            OutputMode,
            PMDAcquisitionMode,
            PMDCalculationMode,
            TemperaturePMD,
            TemperatureIllumination,
            TemperatureCustomer,
            ChipsizeColumns             = 22,
            ChipsizeRows,
            TimeStampIncrement,
            TriggerMode,
            SequencePllSelect           = 96,
            SequenceIntTime,
            SequenceModFrequency,
            SequenceFrameTime,
            SequenceGrayscalePhaseShift,
            SequenceNoPhases,
            SequenceDistanceOffset,
            SequenceAmplitudeMin,
            SequenceSaturation,
            SequenceSymmetry,
            FrameCounter,
            TimeStamp,
            MiraTimeout,
            O1Distance
        }

        public struct SourceDataSTRUCT
        {
            public PMDDataDescription descr;
            public uint[] sourceData;
        }


        public struct ContainerSTRUCT
        {
            public SortedList<ContainerHeaderENUM, UInt32> header;
            public List<double> data;
        }

        public enum AddressRegistersPmdSdk
        {
            PmdStatus = 0x0,
            NoSerialCustomer,
            NoReleaseDate,
            NoFramebytes,
            NoRows,
            NoColumns,
            SetRoi,
            RoiColumnBegin,
            RoiColumnEnd,
            RoiRowBegin,
            RoiRowEnd,
            SequenceLength,
            SequenceCombine,
            OutputMode,
            PmdAcquisitionMode,
            PmdCalculationMode,
            TemperaturePmd,
            TemperatureIllumination,
            TemperatureCustomer,
            UpdateCameraData,
            UpdateFlashData,
            FlashMagic,
            ChipsizeColumns,
            ChipsizeRows,
            TimeStampIncrement,
            TriggerMode,
            TemperatureCompensation,

            ModLedEnable = 0x30,

            Sequence0PllSelect = 0x80,
            Sequence0IntTime,
            Sequence0ModFrequency,
            Sequence0FrameTime,
            Sequence0GrayscalePhaseShift,
            Sequence0NoPhases,
            Sequence0DistanceOffset,
            Sequence0AmplitudeMin,
            Sequence0Saturation,
            Sequence0Symmetry,
            Sequence1PllSelect,
            Sequence1IntTime,
            Sequence1ModFrequency,
            Sequence1FrameTime,
            Sequence1GrayscalePhaseShift,
            Sequence1NoPhases,
            Sequence1DistanceOffset,
            Sequence1AmplitudeMin,
            Sequence1Saturation,
            Sequence1Symmetry,
            Sequence2PllSelect,
            Sequence2IntTime,
            Sequence2ModFrequency,
            Sequence2FrameTime,
            Sequence2GrayscalePhaseShift,
            Sequence2NoPhases,
            Sequence2DistanceOffset,
            Sequence2AmplitudeMin,
            Sequence2Saturation,
            Sequence2Symmetry,
            Sequence3PllSelect,
            Sequence3IntTime,
            Sequence3ModFrequency,
            Sequence3FrameTime,
            Sequence3GrayscalePhaseShift,
            Sequence3NoPhases,
            Sequence3DistanceOffset,
            Sequence3AmplitudeMin,
            Sequence3Saturation,
            Sequence3Symmetry,
            Sequence4PllSelect,
            Sequence4IntTime,
            Sequence4ModFrequency,
            Sequence4FrameTime,
            Sequence4GrayscalePhaseShift,
            Sequence4NoPhases,
            Sequence4DistanceOffset,
            Sequence4AmplitudeMin,
            Sequence4Saturation,
            Sequence4Symmetry,
            Sequence5PllSelect,
            Sequence5IntTime,
            Sequence5ModFrequency,
            Sequence5FrameTime,
            Sequence5GrayscalePhaseShift,
            Sequence5NoPhases,
            Sequence5DistanceOffset,
            Sequence5AmplitudeMin,
            Sequence5Saturation,
            Sequence5Symmetry,
            Sequence6PllSelect,
            Sequence6IntTime,
            Sequence6ModFrequency,
            Sequence6FrameTime,
            Sequence6GrayscalePhaseShift,
            Sequence6NoPhases,
            Sequence6DistanceOffset,
            Sequence6AmplitudeMin,
            Sequence6Saturation,
            Sequence6Symmetry,
            Sequence7PllSelect,
            Sequence7IntTime,
            Sequence7ModFrequency,
            Sequence7FrameTime,
            Sequence7GrayscalePhaseShift,
            Sequence7NoPhases,
            Sequence7DistanceOffset,
            Sequence7AmplitudeMin,
            Sequence7Saturation,
            Sequence7Symmetry,
            Sequence8PllSelect,
            Sequence8IntTime,
            Sequence8ModFrequency,
            Sequence8FrameTime,
            Sequence8GrayscalePhaseShift,
            Sequence8NoPhases,
            Sequence8DistanceOffset,
            Sequence8AmplitudeMin,
            Sequence8Saturation,
            Sequence8Symmetry,
            Sequence9PllSelect,
            Sequence9IntTime,
            Sequence9ModFrequency,
            Sequence9FrameTime,
            Sequence9GrayscalePhaseShift,
            Sequence9NoPhases,
            Sequence9DistanceOffset,
            Sequence9AmplitudeMin,
            Sequence9Saturation,
            Sequence9Symmetry,
            Sequence10PllSelect,
            Sequence10IntTime,
            Sequence10ModFrequency,
            Sequence10FrameTime,
            Sequence10GrayscalePhaseShift,
            Sequence10NoPhases,
            Sequence10DistanceOffset,
            Sequence10AmplitudeMin,
            Sequence10Saturation,
            Sequence10Symmetry,
            Sequence11PllSelect,
            Sequence11IntTime,
            Sequence11ModFrequency,
            Sequence11FrameTime,
            Sequence11GrayscalePhaseShift,
            Sequence11NoPhases,
            Sequence11DistanceOffset,
            Sequence11AmplitudeMin,
            Sequence11Saturation,
            Sequence11Symmetry,

            PhysicalAddressBoundary = 0x100,

            // virtual registers for functions not represented in PmmdSdk dll
            ImageDataFormat,
            FilterConfig,
            WriteConfigToFlash,
            DisablelntegrationTimeCheck,
            LoadCalibrationData,
            IsCalibrationDataLoaded,
            SaveCalibrationData,
            LoadRegisterMap,
            WriteRegisterMap,
            LoadFirmware,
            WriteFirmwareToFlash,

            LastElement,
        }

        public enum ImageDataFormatModes
        {
            DistAmp = 0x00,
            DistAmpConf = 0x08,
            Raw4Phases = 0x10,
            XYZcoordinates = 0x18,
            XYZamp = 0x20,
            XYZcolor = 0x28,
            DistAmpSourceData = 0x30,
            Phases090180270 = 0x38,
            Phases270180900 = 0x40,
            XYZdist = 0x48,
        }


    }
}
